
//*******************************************************************************************************//
//*************************************        INCLUDES          ****************************************//
//*******************************************************************************************************//
#include "F28x_Project.h"
#include "DCLCLA.h"
#include "cla_ex1_background_task_shared.h"
#include "math.h"


#include "driverlib.h"
#include "device.h"
//*******************************************************************************************************//
//*************************************        DEFINES          *****************************************//
//*******************************************************************************************************//

//*************************________Maximum Dead Band values________************************//
#define EPWM2_MAX_DB 0x03FF
#define EPWM2_MIN_DB 0

//*************************________DAC Config Defines________************************//
#define REFERENCE_VDAC 0
#define REFERENCE_VREF 1
#define REFERENCE REFERENCE_VDAC
#define DACA 1
#define DACB 2

//*******************************************************************************************************//
//******************************         SHARED_CLA_VARIABLES           *********************************//
//*******************************************************************************************************//


//*************************________CONTROL________************************//
#pragma DATA_SECTION(Control_PI, "Cla1DataRam")
DCL_PI_CLA Control_PI = PI_CLA_DEFAULTS;

#pragma DATA_SECTION(uk_CLA_2_CPU, "Cla1ToCpuMsgRAM");
float32_t uk_CLA_2_CPU;


//*************************________PWM________************************//
#pragma DATA_SECTION(Duty_CLA_2_CPU, "Cla1ToCpuMsgRAM");
float Duty_CLA_2_CPU;

#pragma DATA_SECTION(Period_CLA_2_CPU, "Cla1ToCpuMsgRAM");
uint16_t Period_CLA_2_CPU;


//*************************________Linker Defined variables________************************//
extern uint32_t Cla1ProgRunStart, Cla1ProgLoadStart, Cla1ProgLoadSize;
extern uint32_t Cla1ConstRunStart, Cla1ConstLoadStart, Cla1ConstLoadSize;


//*******************************************************************************************************//
//********************************        VARIABLES GLOBALES          ***********************************//
//*******************************************************************************************************//
float Accion_PI = 0.0;
uint16_t periodo = 250;
float duty = 0.5;

float SENO;
uint16_t SENO_DAC;
float Precision = 360/(0.25);
float Amplitud = 0.8;
uint16_t Counter = 0;

//*******************************************************************************************************//
//*****************************         DECLARACIONES ESTRUCTURAS           *****************************//
//*******************************************************************************************************//
volatile struct DAC_REGS* DAC_PTR[3] = {0x0,&DacaRegs,&DacbRegs};


//*******************************************************************************************************//
//*******************************        FUNCTION PROTOTYPES          ***********************************//
//*******************************************************************************************************//
void init_ADC_MODULE(void);
void init_ADC_SOC(void);
__interrupt void adcA1ISR(void);
//__attribute__((interrupt))  void cla1ISR1 (void);
//__attribute__((interrupt))  void cla1ISR2 (void);
void init_ePWM1_INTERRUPT(void);

void init_ePWM2_GPIO(void);
void Config_ePWM2_OUTPUT(uint16_t periodo, float duty);
void init_ePWM3_GPIO(void);
void Config_ePWM3_OUTPUT(uint16_t periodo, float duty);

void Twiddle_LEDs(void);
void Config_LEDS(void);

void Parametros_Control_PI(void);

void initCLA(void);

__interrupt void cpuTimer0ISR(void);
void configureDAC(uint16_t dac_num);

//*******************************************************************************************************//
//*********************************         MAIN FUNCTION           *************************************//
//*******************************************************************************************************//
void main(void){

    //*************************________Inicializaciones y Configuraciones________************************//
    InitSysCtrl(); // Initialize device clock and peripherals
    InitGpio(); // Initialize GPIO
    GPIO_SetupPinMux(31U, GPIO_MUX_CPU1, 0);
    GPIO_SetupPinOptions(31U, GPIO_OUTPUT, GPIO_PUSHPULL);

    DINT; // Disable CPU interrupts
    InitPieCtrl(); // Initialize the PIE control registers to their default state.
    IER = 0x0000;// Disable CPU interrupts and clear all CPU interrupt flags:
    IFR = 0x0000;
    InitPieVectTable(); // Initialize the PIE vector table with pointers to the shell Interrupt

    // Map the ISR to the ADCA end-of-task interrupt & CLA1TASK interrupt
    EALLOW; // Map ISR functions
    PieVectTable.TIMER0_INT = &cpuTimer0ISR;
    PieVectTable.ADCA1_INT = &adcA1ISR;     // Function for ADCA interrupt 1
    EDIS;

    InitCpuTimers();
    ConfigCpuTimer(&CpuTimer0, 100, 13.89);
    CpuTimer0Regs.TCR.all = 0x4000;

    IER |= M_INT1;
    PieCtrlRegs.PIEIER1.bit.INTx7 = 1;

    //*************************________Funciones Propias________************************//
    initCLA();
    init_ADC_MODULE();
    init_ADC_SOC();
    init_ePWM1_INTERRUPT();

    init_ePWM2_GPIO();
    Config_ePWM2_OUTPUT(periodo, duty);
    init_ePWM3_GPIO();
    Config_ePWM3_OUTPUT(periodo, duty);

    Parametros_Control_PI();
    configureDAC(DACA);

    SCI_setConfig(SCIA_BASE, 25000000, 9600, (SCI_CONFIG_WLEN_8 |
                                                SCI_CONFIG_STOP_ONE |
                                                SCI_CONFIG_PAR_NONE));

    //*************************________Inicializaciones y Configuraciones________************************//
    EINT; // Enable Global interrupt INTM
    ERTM; // Enable Global realtime interrupt DBGM

    PieCtrlRegs.PIEIER1.bit.INTx1 = 1; // Enable PIE interrupt
    EALLOW;
    CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 1; // Sync ePWM



    //*************************________LOOP________************************//
    while(1){
        EPwm1Regs.ETSEL.bit.SOCAEN = 1;    // 1 = Enable EPWM1SOCA (pg. 214/270)
        EPwm1Regs.TBCTL.bit.CTRMODE = 0;   // TBCTR in Un-Freeze mode (pg. 127/270)
        Accion_PI = uk_CLA_2_CPU;
    }
}


//*******************************************************************************************************//
//***************         initADC => Function to configure and power up ADCA           ******************//
//*******************************************************************************************************//
void init_ADC_MODULE(void){

    SetVREF(ADC_ADCA, ADC_INTERNAL, ADC_VREF3P3); //Usar tensin de referencia interna a 3.3V

    EALLOW;
    //ADC Control 2 Register//
    AdcaRegs.ADCCTL2.bit.PRESCALE = 6; //PRESCALE = 110 (6) --> Input CLK / 4 (pag. 40/180)
    AdcaRegs.ADCCTL1.bit.INTPULSEPOS = 1; //ADC Interrupt Pulse Position --> 1 = Interrup. EOC (pag. 39/180)
    AdcaRegs.ADCCTL1.bit.ADCPWDNZ = 1; //ADC Power Down --> 1 = Power-Up Analog Circuits (pag. 38/180)
    EDIS;

    DELAY_US(1000);
}


//*******************************************************************************************************//
//*****         initADCSOC => Function to configure ADCA's SOC0 to be triggered by ePWM1           ******//
//*******************************************************************************************************//
void init_ADC_SOC(void){ // ADCINA1 --> 30 //// ADCINA6 --> 63

    EALLOW;
    AdcaRegs.ADCSOC0CTL.bit.CHSEL = 1;     // Channel Select; SOC0 --> ADCINA1 (pag. 80/180)
    AdcaRegs.ADCSOC0CTL.bit.ACQPS = 9;     // Acquisition Window; 9 --> 10*SYSCLK cycles (pag. 80/180)
    AdcaRegs.ADCSOC0CTL.bit.TRIGSEL = 5;   // Trigger Source; 5 --> ePWM1, ADCSOCA (pag. 81/180)

    AdcaRegs.ADCSOC1CTL.bit.CHSEL = 6;     // Channel Select; SOC1 --> ADCINA6 (pag. 80/180)
    AdcaRegs.ADCSOC1CTL.bit.ACQPS = 9;     // Acquisition Window; 9 --> 10*SYSCLK cycles (pag. 80/180)
    AdcaRegs.ADCSOC1CTL.bit.TRIGSEL = 5;   // Trigger Source; 5 --> ePWM1, ADCSOCA (pag. 81/180)

    AdcaRegs.ADCINTSEL1N2.bit.INT1SEL = 0; // ADCINT1 EOC Source Select; EOC0 --> triggers ADCINT1 (pag. 51/180)
    AdcaRegs.ADCINTSEL1N2.bit.INT1E = 1;   // ADCINT1 Enable Interrupt (pag. 51/180)
    AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1; // Interrupt Flag Clear Register --> Make sure ADCINT1 flag is cleared (pag. 46/180)
    EDIS;
}


//*******************************************************************************************************//
//*****************         adcA1ISR => ADC A Interrupt 1 ISR + CALCULO DUTY           ******************//
//*******************************************************************************************************//
__interrupt void adcA1ISR(void){

    //*************************________Actualizacion_PWM________************************//
    EPwm2Regs.TBPRD = Period_CLA_2_CPU - 1;
    EPwm2Regs.CMPA.bit.CMPA = Duty_CLA_2_CPU;   // Modulo Comparacin A (TBCTR = CMPA) --> Duty 50% (CONFIGURACION ePWM)
    EPwm2Regs.CMPB.bit.CMPB = Duty_CLA_2_CPU;   // Modulo Comparacin B (TBCTR = CMPB) --> Duty 50% (CONFIGURACION ePWM)

    //*************************________ACK_ADC________************************//
    //"ACK" y "CLEAN" interrupcin (pg. 16/180)
    AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1; // Clear the interrupt flag
    if(1 == AdcaRegs.ADCINTOVF.bit.ADCINT1){ // Check if interrupt overflow has occurred
        AdcaRegs.ADCINTOVFCLR.bit.ADCINT1 = 1; //clear INT1 overflow flag
        AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1; //clear INT1 flag
    }

    PieCtrlRegs.PIEACK.all = PIEACK_GROUP1; // Acknowledge the interrupt
}


//*******************************************************************************************************//
//***********         initEPWM => Function to configure ePWM1 to generate the SOC           *************//
//*******************************************************************************************************//
void init_ePWM1_INTERRUPT(void){

    EALLOW;
    EPwm1Regs.ETSEL.bit.SOCAEN = 0;     // 0 = Disable EPWM1SOCA (pg. 214/270)
    EPwm1Regs.ETSEL.bit.SOCASEL = 4;    // Enable (SOC) TBCTR = CMPA when CTR is count-up (pg. 215/270)
    EPwm1Regs.ETPS.bit.SOCAPRD = 1;     // Generate the EPWM1SOCA pulse on the 1st Event (pg. 218/270)

    EPwm1Regs.CMPA.bit.CMPA = 0x0800;   // Modulo Comparacin A (TBCTR = CMPA = 2048 counts) (pg. 182/270)
    EPwm1Regs.TBPRD = 0x1000;           // Set TBCTR "Time-Base Counter" to 4096 counts (pg. 182/270)

    EPwm1Regs.TBCTL.bit.CTRMODE = 3;    // TBCTR in Freeze mode (pg. 127/270)
    EDIS;
}


//*******************************************************************************************************//
//*************         ePWM2_Output_Config => Function CONFIGURACION PWM OUTPUT           **************//
//*******************************************************************************************************//
void Config_ePWM2_OUTPUT(uint16_t periodo, float duty){

    EPwm2Regs.TBCTL.bit.PRDLD = TB_SHADOW;  // Set Shadow of TBCTR (pg. 126/270)
    EPwm2Regs.TBPRD = periodo - 1;   // Frecuencia inicial del PWM (f=1/periodo) TBCTR "Time-Base Counter" (pg. 182/270)
    EPwm2Regs.CMPA.bit.CMPA = duty;   //  Modulo Comparacin A (TBCTR = CMPA = Duty) (pg. 182/270)
    EPwm2Regs.CMPB.bit.CMPB = duty;   //  Modulo Comparacin B (TBCTR = CMPA = Duty) (pg. 182/270)

    EPwm2Regs.TBPHS.all = 0; //TBCTR Phase & Syncro event is ignored (pg. 179/270)
    EPwm2Regs.TBCTR = 0; //Reset TBCTR del PWM

    EPwm2Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;
    EPwm2Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
    EPwm2Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;
    EPwm2Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;

    EPwm2Regs.TBCTL.bit.CTRMODE = TB_COUNT_UP; //Counter Mode; 00 = TBCTR-Up Mode (pg. 127/270)
    EPwm2Regs.TBCTL.bit.PHSEN = TB_DISABLE; //TBCTR is not loaded from Shadow (pg. 127/270)
    EPwm2Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_DISABLE; //Output Synchro Select; 11 = Disabled (pg. 126/270)
    EPwm2Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1; //High Speed TBCLK preescaler --> Divided by 1 (pg. 126/270)
    EPwm2Regs.TBCTL.bit.CLKDIV = TB_DIV1; //TBCLK preescaler --> Divided by 1 (pg. 126/270)

    //Se generan las seales PWM invertidas
    EPwm2Regs.AQCTLA.bit.ZRO = AQ_SET; //When TBCTR = 0: (AQSet = 3) --> Toggle EPWM2A output (pg. 165/270)
    EPwm2Regs.AQCTLA.bit.CAU = AQ_CLEAR; //When TBCTR = CMPA: (AClear = 1) --> Force Low Output (pg. 165/270)
    EPwm2Regs.AQCTLB.bit.ZRO = AQ_CLEAR; //When TBCTR = 0: (AClear = 1) --> Force Low Output (pg. 165/270)
    EPwm2Regs.AQCTLB.bit.CBU = AQ_SET; //When TBCTR = CMPB: (AQSet = 3) --> Toggle EPWM1B output (pg. 165/270)

    EPwm2Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE; //DBM is fully enabled (i.e. both RED and FED active) (pg. 1946/2773)
    EPwm2Regs.DBCTL.bit.POLSEL = DB_ACTV_HI; //Active high (AH) mode. Neither EPWMxA nor EPWMxB is inverted (default) (pg. 1946/2773)
    EPwm2Regs.DBCTL.bit.IN_MODE = DBA_RED_DBB_FED; //EPWMxA In (from the action-qualifier) is the source for risingedge delayed signal (pg. 1946/2773)
    EPwm2Regs.DBRED.all = EPWM2_MIN_DB;
    EPwm2Regs.DBFED.all = EPWM2_MIN_DB;

    // Interrupt where we will change the deadband
    EPwm2Regs.ETSEL.bit.INTSEL = ET_CTR_ZERO;     // Select INT on Zero event
    EPwm2Regs.ETSEL.bit.INTEN = 1;                // Enable INT
    EPwm2Regs.ETPS.bit.INTPRD = ET_3RD;           // Generate INT on 3rd event

    EALLOW;
    CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 1;
    EDIS;
}


//*******************************************************************************************************//
//****************         initGPIO_ePWM2 => Function CONFIGURACION GPIOS ePWM           ****************//
//*******************************************************************************************************//
void init_ePWM2_GPIO(void){ //GPIO_02 (EPWM2A) --> 76 //// GPIO_03 (EPWM2B) --> 75

    EALLOW;
    GpioCtrlRegs.GPAPUD.bit.GPIO2 = 1; //GPIO_02 A Pull-Up Disable (EPWM2A) (pg. 908/2773)
    GpioCtrlRegs.GPAPUD.bit.GPIO3 = 1; //GPIO_03 A Pull-Up Disable (EPWM2B) (pg. 908/2773)

    GpioCtrlRegs.GPAMUX1.bit.GPIO2 = 1; //GPIO A Peripheral Group Mux (GPIO0 to GPIO15) Configure GPIO2 as EPWM2A (pg. 916/2773)
    GpioCtrlRegs.GPAMUX1.bit.GPIO3 = 1; //GPIO A Peripheral Group Mux (GPIO0 to GPIO15) Configure GPIO3 as EPWM2B (pg. 916/2773)
    EDIS;
}


//*******************************************************************************************************//
//*************         ePWM3_Output_Config => Function CONFIGURACION PWM OUTPUT           **************//
//*******************************************************************************************************//
void Config_ePWM3_OUTPUT(uint16_t periodo, float duty){

    EPwm3Regs.TBCTL.bit.PRDLD = TB_SHADOW;  // Set Shadow of TBCTR (pg. 126/270)
    EPwm3Regs.TBPRD = periodo - 1;   // Frecuencia inicial del PWM (f=1/periodo) TBCTR "Time-Base Counter" (pg. 182/270)
    EPwm3Regs.CMPA.bit.CMPA = duty;   //  Modulo Comparacin A (TBCTR = CMPA = Duty) (pg. 182/270)
    EPwm3Regs.CMPB.bit.CMPB = duty;   //  Modulo Comparacin B (TBCTR = CMPA = Duty) (pg. 182/270)

    EPwm3Regs.TBPHS.all = 0; //TBCTR Phase & Syncro event is ignored (pg. 179/270)
    EPwm3Regs.TBCTR = 0; //Reset TBCTR del PWM

    EPwm3Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;
    EPwm3Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;
    EPwm3Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;
    EPwm3Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;

    EPwm3Regs.TBCTL.bit.CTRMODE = TB_COUNT_UP; //Counter Mode; 00 = TBCTR-Up Mode (pg. 127/270)
    EPwm3Regs.TBCTL.bit.PHSEN = TB_DISABLE; //TBCTR is not loaded from Shadow (pg. 127/270)
    EPwm3Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_DISABLE; //Output Synchro Select; 11 = Disabled (pg. 126/270)
    EPwm3Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1; //High Speed TBCLK preescaler --> Divided by 1 (pg. 126/270)
    EPwm3Regs.TBCTL.bit.CLKDIV = TB_DIV1; //TBCLK preescaler --> Divided by 1 (pg. 126/270)

    //Se generan las seales PWM invertidas
    EPwm3Regs.AQCTLA.bit.ZRO = AQ_SET; //When TBCTR = 0: (AQSet = 3) --> Toggle EPWM2A output (pg. 165/270)
    EPwm3Regs.AQCTLA.bit.CAU = AQ_CLEAR; //When TBCTR = CMPA: (AClear = 1) --> Force Low Output (pg. 165/270)
    EPwm3Regs.AQCTLB.bit.ZRO = AQ_CLEAR; //When TBCTR = 0: (AClear = 1) --> Force Low Output (pg. 165/270)
    EPwm3Regs.AQCTLB.bit.CBU = AQ_SET; //When TBCTR = CMPB: (AQSet = 3) --> Toggle EPWM1B output (pg. 165/270)

    EPwm3Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE; //DBM is fully enabled (i.e. both RED and FED active) (pg. 1946/2773)
    EPwm3Regs.DBCTL.bit.POLSEL = DB_ACTV_HI; //Active high (AH) mode. Neither EPWMxA nor EPWMxB is inverted (default) (pg. 1946/2773)
    EPwm3Regs.DBCTL.bit.IN_MODE = DBA_RED_DBB_FED; //EPWMxA In (from the action-qualifier) is the source for risingedge delayed signal (pg. 1946/2773)
    EPwm3Regs.DBRED.all = EPWM2_MIN_DB;
    EPwm3Regs.DBFED.all = EPWM2_MIN_DB;

    // Interrupt where we will change the deadband
    EPwm3Regs.ETSEL.bit.INTSEL = ET_CTR_ZERO;     // Select INT on Zero event
    EPwm3Regs.ETSEL.bit.INTEN = 1;                // Enable INT
    EPwm3Regs.ETPS.bit.INTPRD = ET_3RD;           // Generate INT on 3rd event

    EALLOW;
    CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 1;
    EDIS;
}


//*******************************************************************************************************//
//****************         initGPIO_ePWM3 => Function CONFIGURACION GPIOS ePWM           ****************//
//*******************************************************************************************************//
void init_ePWM3_GPIO(void){ //GPIO_04 (EPWM3A) --> 36 //// GPIO_05 (EPWM3B) --> 35

    EALLOW;
    GpioCtrlRegs.GPAPUD.bit.GPIO4 = 1; //GPIO_04 A Pull-Up Disable (EPWM2A) (pg. 908/2773)
    GpioCtrlRegs.GPAPUD.bit.GPIO5 = 1; //GPIO_05 A Pull-Up Disable (EPWM2B) (pg. 908/2773)

    GpioCtrlRegs.GPAMUX1.bit.GPIO4 = 1; //GPIO A Peripheral Group Mux (GPIO0 to GPIO15) Configure GPIO4 as EPWM3A (pg. 916/2773)
    GpioCtrlRegs.GPAMUX1.bit.GPIO5 = 1; //GPIO A Peripheral Group Mux (GPIO0 to GPIO15) Configure GPIO5 as EPWM3B (pg. 916/2773)
    EDIS;
}


//*******************************************************************************************************//
//*****************         Config_LEDS => Function que configura los LEDS           ********************//
//*******************************************************************************************************//
//void Config_LEDS (void){
//    EALLOW;
//       // GpioCtrlRegs.GPAAMSEL.bit.GPIO23 = 0; //GPIO_23 --> 0 = Digital Mode (pg. 914/2773)
//
//        GpioCtrlRegs.GPADIR.bit.GPIO23 = 1; //GPIO_Direction: 1 = OUTPUT / 0 = INPUT (pg. 906/2773)
//        GpioCtrlRegs.GPBDIR.bit.GPIO34 = 1; //GPIO_Direction: 1 = OUTPUT / 0 = INPUT (pg. 906/2773)
//
//        GpioDataRegs.GPADAT.bit.GPIO23 = 0; //GPIO_A_DATA: 1 = Low Output Pin (LED OFF) (pg. 968/2773)
//        GpioDataRegs.GPBDAT.bit.GPIO34 = 0; //GPIO_A_DATA: 1 = Low Output Pin (LED OFF) (pg. 968/2773)
//    EDIS;
//}


//*******************************************************************************************************//
//************         Twiddle_LEDs => Function de Blink alternativo de los LEDs           **************//
//*******************************************************************************************************//
//void Twiddle_LEDs (void){
//    LEDs_CPU_2_CLA = 0;
//        GpioDataRegs.GPADAT.bit.GPIO23 = Encendido_CLA_2_CPU;
//        GpioDataRegs.GPBDAT.bit.GPIO34 = Encendido_CLA_2_CPU;
//        DELAY_US(65534);
//        DELAY_US(65534);
//    LEDs_CPU_2_CLA = 1;
//        GpioDataRegs.GPADAT.bit.GPIO23 = Encendido_CLA_2_CPU;
//        GpioDataRegs.GPBDAT.bit.GPIO34 = Encendido_CLA_2_CPU;
//    DELAY_US(65534);
//    DELAY_US(65534);
//}


//*******************************************************************************************************//
//*******************         Parametros_Control_PI => Parmetros Control PI           ******************//
//*******************************************************************************************************//
void Parametros_Control_PI(void){
    Control_PI.Kp = 0.623f; //Constante proporcional
    Control_PI.Ki = 0.12f; //Constante integral
    Control_PI.Umax = 400.0f; //Valor de Saturacion Mxima
    Control_PI.Umin = -400.0f; //Valor de Saturacion Mnima
}


//*******************************************************************************************************//
//*******************         CONFIGURACION DEL Digital-To-Analog-Converter            ******************//
//*******************************************************************************************************//
void configureDAC(uint16_t dac_num)
{
    EALLOW;
    DAC_PTR[dac_num]->DACCTL.bit.DACREFSEL = REFERENCE;
    DAC_PTR[dac_num]->DACOUTEN.bit.DACOUTEN = 1;
    DAC_PTR[dac_num]->DACVALS.all = 0;
    DELAY_US(10); // Delay for buffered DAC to power up
    EDIS;
}


//*******************************************************************************************************//
//***************************         TIM_1 INTERRUPT --> SINE WAVE            **************************//
//*******************************************************************************************************//
__interrupt void cpuTimer0ISR(void)
{
    CpuTimer0.InterruptCount++;
    if(Counter < 1440){
        Counter += 1;
    }else{
        Counter = 0;
    }
    SENO = Amplitud * sinf(2*M_PI*(Counter/Precision));
    SENO_DAC = ((4095-2048)/Amplitud)*(SENO) + 2048;
    DAC_PTR[DACA]->DACVALS.all = SENO_DAC;

    PieCtrlRegs.PIEACK.all = PIEACK_GROUP1;
}

//-----------------------------------------------------------------------------
//
// CLA Initialization
//
// Description: This function will
// - copy over code and const from flash to CLA program and data ram
//   respectively
// - Initialize the task vectors (MVECTx)
// - setup each task's trigger
// - enable each individual task
// - map program and data spaces to the CLA
// - run any one-time initialization task
// Please note that the CLA can only run code and access data that is in RAM.
// the user must assign constants (tables) to FLASH, and copy them over to
// RAM at run-time. They must be copied to a RAM that lies in the address space
// of the CLA, and prior to giving the CLA control over that space
//
//-----------------------------------------------------------------------------
void initCLA(void){

    EALLOW;

    // CLA Program will reside in RAMLS0 and data in RAMLS1, RAMLS2
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS0 = 1U;
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS1 = 0U;
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS2 = 0U;
    // Select CLA as the master of RAMLS0, RAMSL1, RAMLS2
    MemCfgRegs.LSxMSEL.bit.MSEL_LS0 = 1U;
    MemCfgRegs.LSxMSEL.bit.MSEL_LS1 = 1U;
    MemCfgRegs.LSxMSEL.bit.MSEL_LS2 = 1U;

#pragma diag_suppress=770

    // Assign the task vectors and set the triggers for task 1 and 7
    Cla1Regs.MVECT1 = (uint16_t)&Cla1Task1;
    DmaClaSrcSelRegs.CLA1TASKSRCSEL1.bit.TASK1 = 1U; //ADCA1
    Cla1Regs.MIER.bit.INT1 = 1U;

//    Cla1Regs.MVECT1 = (uint16_t)&Cla1Task2;
//    DmaClaSrcSelRegs.CLA1TASKSRCSEL1.bit.TASK2 = 68U; //TINT0
//    Cla1Regs.MIER.bit.INT2 = 1U;

    EDIS;
}

//-----------------------------------------------------------------------------
//
// CLA Task 1 End-of-Task Interrupt Service Routine
//-----------------------------------------------------------------------------
#ifdef __cplusplus
#pragma CODE_SECTION(".TI.ramfunc");
#else
#pragma CODE_SECTION(cla1ISR1, ".TI.ramfunc")
#endif
//__attribute__((interrupt))  void cla1ISR1 (){
//}
//__attribute__((interrupt))  void cla1ISR2 (){
//}


